<?php

namespace App\Http\Controllers\Admin;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Maatwebsite\Excel\Facades\Excel;
use App\Imports\LectureImport;
use App\Category;
use App\Discipline;
use App\Institute;
use App\Instructor;
use App\Lecture;
use App\Level;
use Carbon\Carbon;
use Validator;
use Str;
use Auth;
use App\Exports\DataExport;

class LectureController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $results = Lecture::with(['lectureInstructor', 'lectureCategory', 'lectureDiscipline', 'lectureInstitute', 'lectureLevel'])
            ->when(request('title'), function ($query) {
                $searchTitle = request()->get('title');
                $query->where('title', 'LIKE', "%$searchTitle%");
            })
            ->when(request('discipline'), function ($query) {
                $query->where('discipline', request()->get('discipline'));
            })
            ->when(request('instructor'), function ($query) {
                $query->whereHas('lectureInstructor', function ($sub_query) {
                    $sub_query->where('title', 'LIKE', "%" . request('instructor') . "%");
                });
            })
            ->when(request('institute'), function ($query) {
                $query->where('institute', request()->get('institute'));
            })
            ->when(request('category'), function ($query) {
                $query->whereHas('lectureCategory', function ($sub_query) {
                    $sub_query->where('title', 'LIKE', "%" . request('category') . "%");
                });
            })
            ->when(request('level'), function ($query) {
                $query->where('level', request()->get('level'));
            })
            ->where('deleted', 0)->orderBy('id', 'DESC')->paginate($request->per_page ?? 10 );
        $instructors = Instructor::where(['status' => 1,'deleted'=> 0])->get();
        $institutes = Institute::where(['status' => 1,'deleted'=> 0])->get();
        $categorys = Category::where(['status' => 1,'deleted'=> 0])->get();
        $disciplines = Discipline::where(['status' => 1,'deleted'=> 0])->get();
        $levels = Level::where(['status' => 1,'deleted'=> 0])->get();
        return view('admin.lecture.index', compact('results', 'instructors', 'institutes', 'categorys', 'disciplines', 'levels'));


//        return view('admin.lecture.index', compact('results'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $instructors = Instructor::where(['status' => 1,'deleted'=> 0])->get();
        $institutes = Institute::where(['status' => 1,'deleted'=> 0])->get();
        $categorys = Category::where(['status' => 1,'deleted'=> 0])->get();
        $disciplines = Discipline::where(['status' => 1,'deleted'=> 0])->get();
        $levels = Level::where(['status' => 1,'deleted'=> 0])->get();
        return view('admin.lecture.form', compact('instructors', 'institutes', 'categorys', 'disciplines', 'levels'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|unique:lecture,title'
        ]);
        $fileName = null;
        if (request()->hasFile('course_feature_image')) {
            $file = request()->file('course_feature_image');
            $fileName = '/course_feature_image/' . md5($file->getClientOriginalName()) . time() . "." . $file->getClientOriginalExtension();
            $file->move('./uploads/course_feature_image', $fileName);
        }
        $data = $request->except('_token');
        $data['course_feature_image'] = $fileName;
        $data['status'] = '1';
        $data['created_by'] = Auth::id();
        $data['updated_by'] = Auth::id();
        Lecture::create($data);
        return redirect()->to(route('lecture.index'));
    }

    /**
     * Display the specified resource.
     *
     * @param int $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param int $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $row = Lecture::find($id);
        $instructors = Instructor::where(['status' => 1,'deleted'=> 0])->get();
        $institutes = Institute::where(['status' => 1,'deleted'=> 0])->get();
        $categorys = Category::where(['status' => 1,'deleted'=> 0])->get();
        $disciplines = Discipline::where(['status' => 1,'deleted'=> 0])->get();
        $levels = Level::where(['status' => 1,'deleted'=> 0])->get();
        return view('admin.lecture.form', compact('row', 'instructors', 'institutes', 'categorys', 'disciplines', 'levels'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     * @param int $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $request->validate(['title' => 'required']);
        $fileName = null;
        $row = Lecture::find($id);
        $currentImage = $row->course_feature_image;
        if (request()->hasFile('course_feature_image')) {
            $file = request()->file('course_feature_image');
            $fileName = '/course_feature_image/' . md5($file->getClientOriginalName()) . time() . "." . $file->getClientOriginalExtension();
            $file->move('./uploads/course_feature_image', $fileName);
        }
        $data = $request->all();
        $data['updated_by'] = Auth::id();
        $data['course_feature_image'] = ($fileName) ? $fileName : $currentImage;
        $row->update($data);
        if ($fileName)
            File::delete('./uploads' . $currentImage);
        return redirect()->to(route('lecture.index'));
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param int $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Request $request, $id)
    {
        if ($request->ajax()) {
            $row = Lecture::find($id);
            $row->deleted = 1;
            $row->deleted_at = Carbon::now();
            $row->deleted_by = Auth::id();
            $row->save();
            return 'true';
        }
        return redirect()->back();
    }

    public function status(Request $request, $id)
    {
        if ($request->ajax()) {
            $row = Lecture::find($id);
            $newStatus = ($row->status == '0') ? '1' : '0';
            $row->update(['status' => $newStatus]);
            return $newStatus;
        }
        return redirect()->back();
    }

    public function import(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'file' => 'required|mimes:xls,xlsx'
        ]);
        if ($validator->fails()) {
            return redirect()->back()->with('error', $validator->errors()->first());
        } else {
            Excel::import(new LectureImport(), $request->file('file'));
            return redirect()->back()->with('success', 'File Uploaded Successfully!');
        }

    }

    public function activeAllStatus(Request $request)
    {
        if ($request->ajax()) {
            foreach ($request->selectAll as $key => $value) {
                $row = Lecture::find($value);
                $row->status = 1;
                $row->save();
            }
            return 'true';
        }
        return redirect()->back();
    }

    public function deactiveAllStatus(Request $request)
    {
        if ($request->ajax()) {
            foreach ($request->selectAll as $key => $value) {
                $row = Lecture::find($value);
                $row->status = 0;
                $row->save();
            }
            return 'true';
        }
        return redirect()->back();
    }

    public function deleteAll(Request $request)
    {
        if ($request->ajax()) {
            foreach ($request->selectAll as $key => $value) {
                $row = Lecture::find($value);
                $row->deleted = 1;
                $row->deleted_at = Carbon::now();
                $row->deleted_by = Auth::id();
                $row->save();
            }
            return 'true';
        }
        return redirect()->back();
    }

    public function export(Request $request)
    {
        $export_fields = [
            ["checked" => true, "key" => "id", "name" => "ID"],
            ["checked" => true, "key" => "title", "name" => "Title"],
            ["checked" => true, "key" => "title", "name" => "Instructor", "relation" => "lectureInstructor", "hasMany" => false],
            ["checked" => true, "key" => "title", "name" => "Course", "relation" => "lectureCategory", "hasMany" => false],
            ["checked" => true, "key" => "title", "name" => "Discipline", "relation" => "lectureDiscipline", "hasMany" => false],
            ["checked" => true, "key" => "title", "name" => "Institute", "relation" => "lectureInstitute", "hasMany" => false],
            ["checked" => true, "key" => "title", "name" => "Level", "relation" => "lectureLevel", "hasMany" => false],
            ["checked" => true, "key" => "extension", "name" => "Extension"],
            ["checked" => true, "key" => "course_video_link", "name" => "Course Video Link"],
//            ["checked" => true, "key" => "coupon_discount", "name" => "Coupon Discount"],
//            ["checked" => true, "key" => "discount", "name" => "Discount"],
//            ["checked" => true, "key" => "code", "name" => "Code"],
        ];
        $data = Lecture::with(['lectureInstructor', 'lectureCategory', 'lectureDiscipline', 'lectureInstitute', 'lectureLevel'])
            ->when(request('title'), function ($query) {
                $searchTitle = request()->get('title');
                $query->where('title', 'LIKE', "%$searchTitle%");
            })
            ->when(request('discipline'), function ($query) {
                $query->where('discipline', request()->get('discipline'));
            })
            ->when(request('instructor'), function ($query) {
                $query->whereHas('lectureInstructor', function ($sub_query) {
                    $sub_query->where('title', 'LIKE', "%" . request('instructor') . "%");
                });
            })
            ->when(request('institute'), function ($query) {
                $query->where('institute', request()->get('institute'));
            })
            ->when(request('category'), function ($query) {
                $query->whereHas('lectureCategory', function ($sub_query) {
                    $sub_query->where('title', 'LIKE', "%" . request('category') . "%");
                });
            })
            ->when(request('level'), function ($query) {
                $query->where('level', request()->get('level'));
            })
            ->where('deleted', 0)->orderBy('id', 'DESC')->paginate($request->per_page ?? 10 );

        return Excel::download(new DataExport($export_fields,$data), 'lectures'. Carbon::now() .'.csv');
    }
}
